package gov.va.genisis2.bo;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.activiti.engine.ProcessEngine;
import org.activiti.engine.RepositoryService;
import org.activiti.engine.RuntimeService;
import org.activiti.engine.TaskService;
import org.activiti.engine.runtime.ProcessInstance;
import org.activiti.engine.task.Task;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;

import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisServiceException;
import gov.va.genisis2.vo.EmailDetails;

/**
 * The Class ActivitiService.
 */
@Configuration
@ComponentScan(basePackages = { "gov.va.genisis2.*" })
public class ActivitiService {

	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(ActivitiService.class);

	/** The process engine. */
	private ProcessEngine processEngine;

	/**
	 * This method is used to get activitiProcess engine.
	 * 
	 * @return processEngine This returns ProcessEngine.
	 */
	public ProcessEngine getProcessEngine() {
		return processEngine;
	}

	/**
	 * This method is used to set the activitiProcess engine.
	 * 
	 * @param processEngine
	 *            The processEngine.
	 */
	@Autowired
	public void setProcessEngine(ProcessEngine processEngine) {
		this.processEngine = processEngine;
	}

	/**
	 * This method is used to Start work flow process.
	 * 
	 * @param userId
	 *            The userId.
	 * @return String This returns process instance id.
	 */
	public String startProcess(String userId) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Start workflow process");
		}

		/*
		 * Deploy the process
		 */
		RepositoryService repositoryService = null;
		RuntimeService runtimeService = null;
		ProcessInstance processInstance = null;

		try {
			repositoryService = processEngine.getRepositoryService();

			repositoryService.createDeployment().enableDuplicateFiltering().addClasspathResource("Genisis2WorkFlowProcess.bpmn").deploy();

			runtimeService = processEngine.getRuntimeService();

			Map<String, Object> variableMap = new HashMap<>();
			variableMap.put("processStartedBy", userId);
			processInstance = runtimeService.startProcessInstanceByKey("Genisis2WorkFlowProcess", variableMap);
		} catch (Exception ex) {
			LOGGER.error("ActivitiService.startProcess: Unable to start workflow for UserId: " + userId, ex);
			return StringUtils.EMPTY;
		}
		return processInstance.getId();
	}

	/**
	 * This method is used to get the process task id.
	 * 
	 * @param processId
	 *            The processId.
	 * @param userId
	 *            The userId.
	 * @return String This returns process taskId.
	 */
	public String getprocessTaskId(String processId, String userId) {
		String taskId = StringUtils.EMPTY;
		TaskService taskService = null;

		try {
			taskService = processEngine.getTaskService();
			List<Task> tasks = taskService.createTaskQuery().taskAssignee(userId).processInstanceId(processId).list();
			for (Task task : tasks) {
				taskId = task.getId();
				break;
			}
		} catch (Exception ex) {
			LOGGER.error("ActivitiService.getprocessTaskId: Unable to get process Task Id for UserId: " + userId, ex);
			return StringUtils.EMPTY;
		}

		return taskId;
	}

	/**
	 * This method is used to Claim and complete human task.
	 * 
	 * @param processId
	 *            The processId.
	 * @param userId
	 *            The userId.
	 * @param json
	 *            Json object.
	 * @throws GenisisServiceException
	 *             GenisisServiceException
	 */
	public void claimAndCompleteHumanTask(String processId, String userId, JSONObject json) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("Claim and Complete Human Task");
		}

		if (StringUtils.isBlank(processId)) {
			LOGGER.error("Process ID is null, unable to claim and complete Task");
			return;
		} 
		
		TaskService taskService = null;

		try {
			taskService = processEngine.getTaskService();
			List<Task> tasks = taskService.createTaskQuery().processInstanceId(processId).list();
			JSONArray jsonArray = json.getJSONArray("emailList");
			List<EmailDetails> emailList = new ArrayList<>();

			for (int i = 0; i < jsonArray.length(); i++) {
				EmailDetails edetails = new EmailDetails();
				edetails.setTo((String) jsonArray.getJSONObject(i).get("to"));
				edetails.setFrom((String) jsonArray.getJSONObject(i).get("from"));
				edetails.setSubject((String) jsonArray.getJSONObject(i).get("subject"));
				edetails.setBody((String) jsonArray.getJSONObject(i).get("body"));
				if (!StringUtils.isBlank((String) jsonArray.getJSONObject(i).get("cc"))) {
					edetails.setCc((String) jsonArray.getJSONObject(i).get("cc"));
				}
				emailList.add(edetails);
			}

			String requestorId = (String) json.get("requestorId");
			String approverDecision = (String) json.get("approverDecision");
			String approverId = (String) json.get("approverId");
			String fulfillmentDecision = (String) json.get("fulfillmentDecision");
			String resultAcceptanceDecision = (String) json.get("resultAcceptanceDecision");
			String resultDeliverDecision = (String) json.get("resultDeliverDecision");

			for (Task task : tasks) {
				String logMessage = "Claiming  Task: " + task.getId() + " for user:" + userId;
				if (LOGGER.isInfoEnabled()) {
					LOGGER.info(logMessage);
				}
				
				String taskClaimedBy = StringUtils.isBlank(task.getAssignee()) ? userId : task.getAssignee();
				taskService.claim(task.getId(), taskClaimedBy);

				Map<String, Object> variableMap = new HashMap<>();
				variableMap.put("HumanTaskCompletedBy", userId);
				variableMap.put("emailList", emailList);
				variableMap.put("requestorId", requestorId);
				variableMap.put("approverDecision", approverDecision);
				variableMap.put("fulfillmentDecision", fulfillmentDecision);
				variableMap.put("resultAcceptanceDecision", resultAcceptanceDecision);
				variableMap.put("resultDeliverDecision", resultDeliverDecision);
				variableMap.put("approverId", approverId);
				taskService.complete(task.getId(), variableMap);

				if (LOGGER.isInfoEnabled()) {
					LOGGER.info("Completed Task");
				}

				break;
			}
		} catch (Exception ex) {
			LOGGER.error(ErrorEnum.ACTIVITI_EXP_HUMANTASK.getErrorMessage(), ex);
			throw new GenisisServiceException(ErrorEnum.ACTIVITI_EXP_HUMANTASK.getErrorMessage(), ex);
		}
	}
}
